import { supabase } from '@/config/supabase';
import {
  MarketPrice,
  Market,
  SubmitMarketPriceRequest,
  UpdatePricingAccessRequest
} from './types';

export const marketPricesService = {
  // Submit market price (pricing agent or admin)
  async submitMarketPrice(user: any, req: SubmitMarketPriceRequest) {
    // Check permission
    if (!user.can_update_prices && user.role !== 'admin') {
      return { error: { message: 'Not authorized to submit prices.' } };
    }
    // Prevent duplicate for same product/market/day
    const today = new Date().toISOString().slice(0, 10);
    const { data: existing } = await supabase.from('market_prices')
      .select('id')
      .eq('product_id', req.product_id)
      .eq('market_id', req.market_id)
      .gte('submitted_at', today + 'T00:00:00Z')
      .lte('submitted_at', today + 'T23:59:59Z')
      .single();
    if (existing) {
      // Overwrite
      const { data, error } = await supabase.from('market_prices').update({
        price: req.price,
        unit: req.unit,
        submitted_by: user.id,
        submitted_at: new Date().toISOString()
      }).eq('id', existing.id).select().single();
      return { data, error };
    } else {
      // Insert new
      const { data, error } = await supabase.from('market_prices').insert({
        product_id: req.product_id,
        market_id: req.market_id,
        price: req.price,
        unit: req.unit,
        submitted_by: user.id,
        submitted_at: new Date().toISOString()
      }).select().single();
      return { data, error };
    }
  },

  // List market prices (with filters)
  async listMarketPrices(filters: any) {
    let query = supabase.from('market_prices').select('*');
    if (filters.market_id) query = query.eq('market_id', filters.market_id);
    if (filters.product_id) query = query.eq('product_id', filters.product_id);
    if (filters.submitted_by) query = query.eq('submitted_by', filters.submitted_by);
    if (filters.date_range) {
      const [from, to] = filters.date_range;
      query = query.gte('submitted_at', from).lte('submitted_at', to);
    }
    const { data, error } = await query.order('submitted_at', { ascending: false });
    return { data, error };
  },

  // Get price suggestions for vendor
  async getSuggestions(vendor: any) {
    // Use vendor's market preference (assume vendor.market_id)
    const market_id = vendor.market_id;
    if (!market_id) return { error: { message: 'Vendor has no market preference set.' } };
    // Get average price per product in this market
    const { data, error } = await supabase.rpc('get_market_price_suggestions', { market_id });
    // Fallback: aggregate manually if no RPC
    // ...
    return { data, error };
  },

  // Admin: approve/reject pricing agent
  async updatePricingAccess(user_id: string, req: UpdatePricingAccessRequest) {
    const { data, error } = await supabase.from('users').update({ can_update_prices: req.can_update_prices }).eq('id', user_id).select().single();
    return { data, error };
  }
}; 